<?php
// Incluir la configuración de la base de datos y la sesión
// Es CRÍTICO que config.php inicie la sesión con session_start()
// y que NO haya NINGÚN espacio en blanco, línea vacía o texto
// antes de la etiqueta <?php en config.php o en este archivo.
// Si session_start() ya se llama en config.php, no es necesario aquí.
// Si no, descomenta la siguiente línea:
// session_start(); 
require_once 'config.php';

// Verificar si el método de la solicitud es POST
if ($_SERVER["REQUEST_METHOD"] == "POST") {

    // Verificar si el usuario ha iniciado sesión
    if (!isset($_SESSION["loggedin"]) || $_SESSION["loggedin"] !== true) {
        // Si no ha iniciado sesión, redirigir a la página de login con un error codificado
        header("location: login.php?error=Debes iniciar sesión para calificar.");
        exit;
    }

    // Recoger los datos del formulario
    $id_usuario = $_SESSION["id"];
    $nombre_restaurante = trim($_POST['nombre_restaurante']);
    $calificacion = (int)$_POST['calificacion'];
    $comentario = trim($_POST['comentario']);
    $redirect_url = trim($_POST['redirect_url']);
    
    // --- VALIDACIÓN DE SEGURIDAD CRÍTICA para redirect_url ---
    // Para prevenir ataques de Open Redirect, asegúrate de que la URL de redirección
    // sea interna o esté en una lista blanca.
    // Ejemplo básico: verificar que sea una ruta relativa o de tu dominio.
    // Una implementación más robusta podría usar una lista blanca de URLs permitidas.
    if (!empty($redirect_url) && (strpos($redirect_url, '://') !== false || substr($redirect_url, 0, 1) !== '/')) {
        // Si la URL contiene un protocolo o no es una ruta relativa, redirigir a una página segura por defecto
        $redirect_url = 'Index.php'; // O la página principal de tu sitio
    }

    // Validar los datos
    if (empty($nombre_restaurante) || empty($calificacion) || empty($redirect_url)) {
        // Si faltan datos clave, redirigir con un error
        header("location: " . $redirect_url . "?error=" . urlencode("Faltan datos para procesar la calificación."));
        exit;
    }

    if ($calificacion < 1 || $calificacion > 5) {
        // Si la calificación está fuera del rango, redirigir con un error
        header("location: " . $redirect_url . "?error=" . urlencode("La calificación debe estar entre 1 y 5."));
        exit;
    }
    
    // --- LÓGICA DE UPSERT: ACTUALIZAR O INSERTAR ---

    // 1. Comprobar si ya existe una calificación para este usuario y restaurante
    $sql_check = "SELECT id FROM calificaciones WHERE id_usuario = ? AND nombre_restaurante = ?";
    if ($stmt_check = $conn->prepare($sql_check)) {
        $stmt_check->bind_param("is", $id_usuario, $nombre_restaurante);
        $stmt_check->execute();
        $stmt_check->store_result();

        if ($stmt_check->num_rows > 0) {
            // 2. Si existe, ACTUALIZAR la calificación existente
            $sql_update = "UPDATE calificaciones SET calificacion = ?, comentario = ?, fecha = NOW() WHERE id_usuario = ? AND nombre_restaurante = ?";
            if ($stmt_update = $conn->prepare($sql_update)) {
                $stmt_update->bind_param("isss", $calificacion, $comentario, $id_usuario, $nombre_restaurante); // Corregido: 's' para comentario
                if ($stmt_update->execute()) {
                    header("location: " . $redirect_url . "?exito=" . urlencode("¡Tu calificación ha sido actualizada!"));
                } else {
                    error_log("Error al actualizar calificación: " . $stmt_update->error);
                    header("location: " . $redirect_url . "?error=" . urlencode("Hubo un error al actualizar tu calificación."));
                }
                $stmt_update->close();
            }
        } else {
            // 3. Si no existe, INSERTAR una nueva calificación
            $sql_insert = "INSERT INTO calificaciones (id_usuario, nombre_restaurante, calificacion, comentario) VALUES (?, ?, ?, ?)";
            if ($stmt_insert = $conn->prepare($sql_insert)) { // Corregido: 's' para comentario
                $stmt_insert->bind_param("isss", $id_usuario, $nombre_restaurante, $calificacion, $comentario);
                if ($stmt_insert->execute()) {
                    header("location: " . $redirect_url . "?exito=" . urlencode("¡Gracias por tu calificación!"));
                } else {
                    error_log("Error al insertar calificación: " . $stmt_insert->error);
                    header("location: " . $redirect_url . "?error=" . urlencode("Hubo un error al guardar tu calificación."));
                }
                $stmt_insert->close();
            }
        }
        $stmt_check->close();
    } else {
        error_log("Error al preparar la sentencia de verificación de calificación: " . $conn->error);
        header("location: " . $redirect_url . "?error=" . urlencode("Error interno del servidor. Inténtalo de nuevo más tarde."));
    }

    // Cerrar la conexión
    $conn->close();

} else {
    // Si se accede al archivo directamente, redirigir a la página principal
    header("location: Index.php");
    exit;
}
?>