<?php
require 'config.php';

// Función para enviar una respuesta JSON y terminar el script
function json_response($success, $message, $data = []) {
    header('Content-Type: application/json');
    echo json_encode(['success' => $success, 'message' => $message, 'data' => $data]);
    exit();
}


if ($_SERVER["REQUEST_METHOD"] == "POST") {
    // 1. Recoger y limpiar los datos del formulario
    $usuario = trim($_POST['usuario']);
    $email = trim($_POST['email']);
    $password = $_POST['password'];
    $confirm_password = $_POST['confirm_password'];

    // 2. Validaciones del lado del servidor
    if (empty($usuario) || empty($email) || empty($password)) {
        json_response(false, "Todos los campos son obligatorios.");
    }

    if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        json_response(false, "El formato del correo no es válido.");
    }

    if ($password !== $confirm_password) {
        json_response(false, "Las contraseñas no coinciden.");
    }

    if (strlen($password) < 6) {
        json_response(false, "La contraseña debe tener al menos 6 caracteres.");
    }

    // 3. Verificar si el usuario o email ya existen (usando sentencias preparadas para evitar inyección SQL)
    $sql_check = "SELECT id FROM usuarios WHERE nombre_usuario = ? OR email = ?";
    
    // Manejo de errores para la preparación de la sentencia
    if (!($stmt_check = $conn->prepare($sql_check))) {
        // Esto indica un error en la consulta SQL o en la conexión
        error_log("Error al preparar la sentencia de verificación: " . $conn->error);
        $conn->close();
        json_response(false, "Error interno del servidor. Inténtalo de nuevo más tarde.");
    }
    
    $stmt_check->bind_param("ss", $usuario, $email);
    $stmt_check->execute();
    $stmt_check->store_result();

    if ($stmt_check->num_rows > 0) {
        $stmt_check->close(); // Cerrar la sentencia antes de cerrar la conexión y salir
        $conn->close();
        json_response(false, "El nombre de usuario o el correo ya están en uso.");
    }
    $stmt_check->close();

    // 4. Hashear la contraseña (¡NUNCA guardar contraseñas en texto plano!)
    $password_hashed = password_hash($password, PASSWORD_DEFAULT);

    // 5. Insertar el nuevo usuario en la base de datos (usando sentencias preparadas)
    $sql_insert = "INSERT INTO usuarios (nombre_usuario, email, password) VALUES (?, ?, ?)";
    if (!($stmt_insert = $conn->prepare($sql_insert))) {
        error_log("Error al preparar la sentencia de inserción: " . $conn->error);
        $conn->close();
        json_response(false, "Error interno del servidor. Inténtalo de nuevo más tarde.");
    }

    $stmt_insert->bind_param("sss", $usuario, $email, $password_hashed);

    if ($stmt_insert->execute()) {
        // Redirigir al login con un mensaje de éxito
        json_response(true, "Registro completado. Ahora puedes iniciar sesión.");
    } else {
        // Manejar un error inesperado de la base de datos
        error_log("Error al ejecutar la inserción de usuario: " . $stmt_insert->error); // Log del error específico
        json_response(false, "Error al registrar el usuario. Inténtalo de nuevo.");
    }

    $stmt_insert->close();
    $conn->close();
} else {
    // Si alguien intenta acceder a este archivo directamente, lo redirigimos
    header("Location: login.php");
    exit();
}
?>